#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from heat.common.i18n import _
from heat.engine import attributes
from heat.engine import constraints
from heat.engine.hot import parameters
from heat.engine import properties
from heat.engine import resource
from heat.engine import support


class Value(resource.Resource):
    """A resource which exposes its value property as an attribute.

    This is useful for exposing a value that is a simple manipulation
    of other template parameters and/or other resources.
    """

    support_status = support.SupportStatus(version='7.0.0')

    PROPERTIES = (
        VALUE, TYPE,
    ) = (
        'value', constraints.Schema.TYPE,
    )

    ATTRIBUTES = (
        VALUE_ATTR,
    ) = (
        'value',
    )

    properties_schema = {
        VALUE: properties.Schema(
            properties.Schema.ANY,
            _('The expression to generate the "value" attribute.'),
            required=True,
            update_allowed=True,
        ),
        TYPE: properties.Schema(
            properties.Schema.STRING,
            _('The type of the "value" property.'),
            constraints=[constraints.AllowedValues(
                parameters.HOTParamSchema.TYPES)],
            update_allowed=True,
        ),
    }

    attributes_schema = {
        VALUE_ATTR: attributes.Schema(
            _('The value generated by this resource\'s properties "value" '
              'expression, with type determined from the properties "type".'),
        ),
    }

    def _resolve_attribute(self, name):
        props = self.frozen_definition().properties(self.properties_schema,
                                                    self.context)
        if name == self.VALUE_ATTR:
            return props[self.VALUE]

    def handle_create(self):
        self.resource_id_set(self.physical_resource_name())

    def handle_update(self, json_snippet, tmpl_diff, prop_diff):
        # allow update, no replace necessary (do not throw UpdateReplace).
        # the resource properties are updated appropriately in parent class.
        pass

    def reparse(self, *args, **kwargs):
        super(Value, self).reparse(*args, **kwargs)
        value_type = self.properties[self.TYPE]
        if value_type is None:
            # We don't know what type the value is, anything goes
            return

        param_type_map = {
            parameters.HOTParamSchema.STRING: constraints.Schema.STRING,
            parameters.HOTParamSchema.NUMBER: constraints.Schema.NUMBER,
            parameters.HOTParamSchema.LIST: constraints.Schema.LIST,
            parameters.HOTParamSchema.MAP: constraints.Schema.MAP,
            parameters.HOTParamSchema.BOOLEAN: constraints.Schema.BOOLEAN,
        }
        if value_type not in param_type_map:
            # the self.TYPE value is not valid, the error will become
            # apparent appropriately when its constraints are checked.
            return
        # We know the value's type, update value's property schema accordingly
        self.properties.props[self.VALUE] = properties.Property(
            properties.Schema(
                param_type_map[value_type],
                _('The expression to generate the "value" attribute.'),
                required=True,
                update_allowed=True,
            ))


def resource_mapping():
    return {
        'OS::Heat::Value': Value,
    }
