#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import subprocess
import apt
import shutil
import gettext
import os
import re
_ = gettext.gettext
gettext.bindtextdomain("desktop-menu","/usr/share/locale")
gettext.textdomain("desktop-menu")

def pretty_errors(e,typ,details):
    print(f"An error occurred: {e}")
    match typ:
        case 'permission':
            print(f"Processing menu file {details} has failed.")
        case 'link':
            print(f"Processing symlink creation for {details} has failed.")
        case 'diversion':
            print(f"Processing diversion for {details} has failed.")
        case 'divert_removal':
            print(f"Processing diversion removal for {details} has failed.")
        case 'link_removal':
            print(f"Processing symlink removal for {details} has failed.")
        case _:
            print("Unknown error.")
    print("Can't proceed.")
    sys.exit(1)

def chrooted():
# retuns true if script is run in chroot, otherwise false. Needed for antiX ISO building. Only call as root.
    if os.lstat('/').st_ino == os.stat('/proc/1/root').st_ino and os.lstat('/').st_dev == os.stat('/proc/1/root').st_dev:
        # the devicenumber/inode pair of / is the same as that of /sbin/init's
        # root, so we're *not* in a chroot and hence return false.
        return False
    return True


def antix_desktop_files():
# Processing for antiX specific .desktop file replacements
    desktop_files_package = 'antix23-desktop-files'
    # check whether desktop-files package is installed on the system
    aptcache = apt.Cache()
    try:
        if not aptcache[desktop_files_package].is_installed:
            return
    except KeyError:
        return
    # Add diversions for .desktop files of recently installed programs
    # read .desktop filenames present in /usr/share/applications into a dictionary/list
    programs = [i for i in os.listdir('/usr/share/applications') if i.endswith('.desktop')]
    antix = [i for i in os.listdir('/usr/local/lib/antiX/{0}/reservoir'.format(desktop_files_package))]
    originals = [i for i in os.listdir('/usr/local/lib/antiX/{0}/originale'.format(desktop_files_package))]
    diversions = []
    env = os.environ.copy(); env['LANG'] = 'C' # Set LANG to C for consistent output of dpkg-divert in localised systems
    divlist = subprocess.run(['dpkg-divert','--list'], stdout=subprocess.PIPE,check=True,encoding='utf-8',env=env).stdout.splitlines() # fetch all current diversions
    # list comprehension for extraction of antiX diversions
    diversions = [match.group(1) for line in divlist if (match := re.match(rf'^diversion of /usr/share/applications/(..*) to .* by {re.escape(desktop_files_package)}', line))]
    # list comprehension for extraction of new .desktop files
    divert_now = [i for i in programs if i in antix if i not in originals if i not in diversions]
    # gksu throws an error if called by root, causing trouble here, so separate treatment needed
    if os.geteuid() == 0:
        # create new diversions from the listing and create symlinks
        for desktopfile in divert_now:
            try:
                subprocess.run(['dpkg-divert','--package',f'{desktop_files_package}','--divert',f'/usr/local/lib/antiX/{desktop_files_package}/originale/{desktopfile}','--rename',f'/usr/share/applications/{desktopfile}'], check=True, encoding='utf-8')
            except subprocess.CalledProcessError as e:
                pretty_errors(e,'diversion',desktopfile)
            try:
                subprocess.run(['ln','-s',f'/usr/local/lib/antiX/{desktop_files_package}/reservoir/{desktopfile}',f'/usr/share/applications/{desktopfile}'],check=True,encoding='utf-8')
            except subprocess.CalledProcessError as e:
                pretty_errors(e,'link',desktopfile)
    else:
        # create new diversions from the listing and create symlinks
        for desktopfile in divert_now:
            try:
                subprocess.run(['gksu','--sudo-mode','--','dpkg-divert','--package',f'{desktop_files_package}','--divert',f'/usr/local/lib/antiX/{desktop_files_package}/originale/{desktopfile}','--rename',f'/usr/share/applications/{desktopfile}'], check=True, encoding='utf-8')
            except subprocess.CalledProcessError as e:
                pretty_errors(e,'diversion',desktopfile)
            try:
                subprocess.run(['gksu','--sudo-mode','--','ln','-s',f'/usr/local/lib/antiX/{desktop_files_package}/reservoir/{desktopfile}',f'/usr/share/applications/{desktopfile}'],check=True,encoding='utf-8')
            except subprocess.CalledProcessError as e:
                pretty_errors(e,'link',desktopfile)
    # Remove diversions for .desktop files from currently uninstalled programs
    # read in updated originals folder and updated diversions
    originals = [i for i in os.listdir('/usr/local/lib/antiX/{0}/originale'.format(desktop_files_package))]
    divlist = subprocess.run(['dpkg-divert','--list'], stdout=subprocess.PIPE,check=True,encoding='utf-8',env=env).stdout.splitlines()
    # list comprehension for extraction of antiX diversions
    diversions = [match.group(1) for line in divlist if (match := re.match(rf'^diversion of /usr/share/applications/(..*) to .* by {re.escape(desktop_files_package)}', line))]
    remove_diversion = [i for i in diversions if i not in originals]
    if os.geteuid() == 0:
        # delete diversions for recently uninstalled programs, root
        for desktopfile in remove_diversion:
            try:
                subprocess.run(['dpkg-divert','--package',f'{desktop_files_package}','--rename','--remove',f'/usr/share/applications/{desktopfile}'], check=True, encoding='utf-8')
            except subprocess.CalledProcessError as e:
                pretty_errors(e,'divert_removal',desktopfile)
            try:
                subprocess.run(['rm','-f',f'/usr/share/applications/{desktopfile}'], check=True, encoding='utf-8')
            except subprocess.CalledProcessError as e:
                pretty_errors(e,'link_removal',desktopfile)
    else:
        # remove diversions for recently uninstalled programs, users
        for desktopfile in remove_diversion:
            try:
                subprocess.run(['gksu','--sudo-mode','--','dpkg-divert','--package',f'{desktop_files_package}','--rename','--remove',f'/usr/share/applications/{desktopfile}'], check=True, encoding='utf-8')
            except subprocess.CalledProcessError as e:
                pretty_errors(e,'divert_removal',desktopfile)
            try:
                subprocess.run(['gksu','--sudo-mode','--','rm','-f',f'/usr/share/applications/{desktopfile}'], check=True, encoding='utf-8')
            except subprocess.CalledProcessError as e:
                pretty_errors(e,'link_removal',desktopfile)
